function [RecInfo] = LoadExperimentalData(Inputfile, Sheet, NumNames, Nstr_descr,Nnum_descr)
%Load an excel or a comma separated datafile with one header line and data organised in
%records with record descriptors (e.g, experiment, latitude, longitude, year, month, day) 
%For fixed format fiels, one can indicate the number of in the first few columns
%
%Function usage:
% For a csv/txt file with fixed format and 3 text and 4 numerical site
% descriptor columns
% [RecInfo] = LoadExperimentalData(Inputfile, 3, 4)
%
% For a xls file
% [RecInfo] = LoadExperimentalData(Inputfile, Sheet)
% For a csv file with known column names for site descriptors 
% [RecInfo] = LoadExperimentalData(Inputfile, [],{'n_champ' 'quadrat'})
%
%Inputs: 
%     Inputfile: filename of excel or comma separated file including path and extension 
%         Sheet: name of data sheet. Only for excel files
%      NumNames: list of column names describing the record with a numerical
%                value, e.g. field or plot number
%    Nstr_descr: for fixed formats only. Number of columns
%                with text site descriptors (field name etc)
%    Nnum_descr: for fixed formats only. Number of columns
%                with numerical site descriptors (plot number etc)
%
%Outputs: structure with record info
%  elements: Var,
%      INPUTFILE: path and name of inputfile used
%    StringNames: headername of text descriptors of records
%       NumNames: headername of numerical descriptors of records
%       VarNames: names of variables in records
%         String: text descriptors of records
%            Num: numerical descriptors of records
%            Var: value of variables in records
%       NRecords: number of records
%
%Author
%Dr A.G.T Schut 2015, Wageningen University

%constants
INVAL_INDICATOR=-9999;        %Value for missing data
if nargin < 2  
    Sheet =[];
end
if nargin < 3
    NumNames =[];
end
if nargin < 4 
    if isnumeric(Sheet) && isnumeric(NumNames)
        Nstr_descr=Sheet; Sheet=[];
        Nnum_descr=NumNames; NumNames=[];
    else
        Nstr_descr=[];
        Nnum_descr=[];
    end
end


if ~isempty(Nstr_descr) && ~isempty(Nstr_descr) && strcmp(Inputfile(end-2:end),'csv')
%Comma separated data with fixed format
    %=================LOADING RAW DATA=========================================
    n_site_descriptors = Nnum_descr  + Nstr_descr;

    %read headerline
    fid=fopen(Inputfile);line=fgetl(fid);fclose(fid);
    header=strread(line,'%s','delimiter',',');
    %define number of variables
    nvar=length(header)-n_site_descriptors;

    %extract names of variables from the header line
    RecInfo.INPUTFILE=Inputfile;
    RecInfo.StringNames=header(1:Nstr_descr);
    RecInfo.NumNames=header(Nstr_descr+1:n_site_descriptors);
    RecInfo.VarNames=header(n_site_descriptors+1:n_site_descriptors+nvar);

    %Define format how to read the data with strings and floating point numbers
    fstr=[repmat('%s ',1,Nstr_descr) repmat('%f ',1,Nnum_descr) repmat('%f ',1,nvar)];

    %Read in the actual data from the file
    [RecInfo.String{1:Nstr_descr}, RecInfo.Num{1:Nnum_descr}, RecInfo.Var{1:nvar}]= ...
        textread(Inputfile,fstr,'headerlines',1,'delimiter',',');
    %convert cells to one matrix!
    RecInfo.Num=cell2mat(RecInfo.Num);
    RecInfo.Var=cell2mat(RecInfo.Var);

elseif strcmp(Inputfile(end-3:end),'xlsx') || strcmp(Inputfile(end-2:end),'xls') || strcmp(Inputfile(end-2:end),'csv')
    %reading free format files    
    if isempty(Sheet)
        [NUM,TXT,RAW]=xlsread(Inputfile);
    else
        [NUM,TXT,RAW]=xlsread(Inputfile,Sheet);
    end
    
    %extract names of variables from the header line
    RecInfo.INPUTFILE=Inputfile;
    RecInfo.INPUTSHEET=Sheet;
    RecInfo.RAW=RAW;
    %Evaluate text and numerical columns
    nVAR=0;nNUMinfo=0;nTXT=0;
    %find numerical values
    for c=1:size(RAW,2)
        mVal=repmat(NaN,size(RAW,1)-1,1);cNUM=0;
        %convert to characters if possible
        cVal=char(RAW{2:end,c});
        for r=2:size(RAW,1)
            val=RAW{r,c};
            if ~ischar(val) && ~isnan(val)
                mVal(r-1,1)=val;
                cNUM=cNUM+1;
            end
        end

        if cNUM > 0 %at least one numerical value
            %a numerical column with at least one value
            if find(strcmp(RAW{1,c},NumNames))
                nNUMinfo=nNUMinfo+1;
                RecInfo.NumNames{nNUMinfo}=RAW{1,c};
                RecInfo.Num(:,nNUMinfo)=mVal;
            else
                nVAR=nVAR+1;
                RecInfo.VarNames{nVAR}=RAW{1,c};
                RecInfo.Var(:,nVAR)=mVal;
            end
        else
            nTXT=nTXT+1;
            RecInfo.StringNames{nTXT}=RAW{1,c};
            RecInfo.String{nTXT}= cellstr(cVal);
        end
        
    end 
else
    error('only fixed format textfiles, or xls or csv formats supported with one header row');
end

%convert invalid number to a NaN
a=logical(RecInfo.Var==INVAL_INDICATOR);
RecInfo.Var(a)=NaN;
RecInfo.NRecords = size(RecInfo.Var,1);
return
