function [RecInfo] = ExperimentalData(ProjectPath, InputFile, Display)
%[RecInfo] = ExperimentalData(Inputfile,Display)
%Usage: Example script to demonstrate how pre-processing and screening of raw data
%       can be organised. ExperimentalData adds a few very basic
%       calculations and displays a selection of the data.
%INPUTS:
%   ProjectPath: a string describing the root project path containing the
%                default folders Data, Models, Paper, Scripts
%   Inputfile  : a string describing the name of the comma separated textfile. 
%                The included example data is taken from the MSc student project of Chiara Perini, 2015.  
%                The Aquaponics experiment ran in Driel, the Netherlands.
%              : the file must be located in the ...Data\Raw\ folder.
%   Display    : [0/1] for display of a figure
%
%OUTPUTS: 
%   RecInfo  :    Structure containing raw and processed data
%   The data is exported to a comma separated file in the
%           ...Data\Processed\ folder, with the same name as the inputfile, but
%           with _processed.csv added.
%
%Uses LoadExperimentalData function to load a comma separated datafile. with one header line and data organised in
%records with record descriptors (e.g, experiment, latitude, longitude, year, month, day) in the first few columns
%
%
%Example function usage when unpacking the example zipfile at D:\Thesis\
% ProjectPath = 'D:\Thesis\'; 
% Inputfile = 'AQUAPONICS lettuce experiment 2014.csv';
% [RecInfo] = ExperimentalData(ProjectPath,Inputfile, 1)
%
%Dr A.G.T Schut 2015, Wageningen University

PathIn=[ProjectPath 'DATA\RAW\'];
PathOut=[ProjectPath 'DATA\PROCESSED\'];
PathResults=[ProjectPath 'RESULTS\'];
Outputfile=[PathOut InputFile(1:end-4) '_processed.csv'];
OutputFigure=[PathResults InputFile(1:end-4) '_figure']; %without extension!

%Load experimental data file
[RecInfo] = LoadExperimentalData([PathIn InputFile],2,4);

%=================SPECIFIC EXPEMENT CALCULATIONS===========================
%Example using data from the Aquaponics project of MSc project of Chiara Perini, 2015
iYR=find(strcmp(RecInfo.NumNames, 'YEAR'));
iMO=find(strcmp(RecInfo.NumNames, 'MONTH'));
iDAY=find(strcmp(RecInfo.NumNames, 'DAY'));
[DOY]=daynumber(RecInfo.Num(iYR),RecInfo.Num(iMO),RecInfo.Num(iDAY));
iBATCH=find(strcmp(RecInfo.StringNames, 'BATCH'));

%Raft in Aquaponics system (located in Driel) with low and high plant densities
%Add plant density as numerical value
iPD=length(RecInfo.VarNames)+1;
RecInfo.VarNames{iPD}='Plant density (plants/m2 raft)';
RecInfo.Var(logical(strcmp(RecInfo.String{iBATCH},'7DAT')),iPD) =41;  % plants/m2 raft
RecInfo.Var(logical(strcmp(RecInfo.String{iBATCH},'14DAT')),iPD)=41;  % plants/m2 raft
RecInfo.Var(logical(strcmp(RecInfo.String{iBATCH},'21DAT')),iPD)=20.5;% plants/m2 raft
RecInfo.Var(logical(strcmp(RecInfo.String{iBATCH},'28DAT')),iPD)=20.5;% plants/m2 raft

%Determine dry weights corrected for Tare
iTARE=find(strcmp(RecInfo.VarNames, 'Tare (g)'));
iFW=find(strcmp(RecInfo.VarNames, 'Fresh weight (g/plant)'));
iDW=find(strcmp(RecInfo.VarNames, 'Dry weight (g/plant)'));
iLA=find(strcmp(RecInfo.VarNames, 'Leaf area (cm2/plant)'));

iFM=length(RecInfo.VarNames)+1;
RecInfo.VarNames{iFM}='FM (g/plant)';
RecInfo.Var(:,iFM) = RecInfo.Var(:,iFW) - RecInfo.Var(:,iTARE);

iDM=length(RecInfo.VarNames)+1;
RecInfo.VarNames{iDM}='DM (g/plant)';
RecInfo.Var(:,iDM) = RecInfo.Var(:,iDW) - RecInfo.Var(:,iTARE);

%Calculate dry matter contents and LAI
iDMc=length(RecInfo.VarNames)+1;
RecInfo.VarNames{iDMc}='DM content(%)';
RecInfo.Var(:,iDMc) = 100 * RecInfo.Var(:,iDM)./RecInfo.Var(:,iFM);

%Calculate LAI
iLAI=length(RecInfo.VarNames)+1;
RecInfo.VarNames{iLAI}='LAI (m2 leaf/m2 soil)';
RecInfo.Var(:,iLAI) = 0.0001 * RecInfo.Var(:,iLA) .* RecInfo.Var(:,iPD);

%Calculate SLA
iSLA=length(RecInfo.VarNames)+1;
RecInfo.VarNames{iSLA}='SLA (m2 leaf / g DM)';
RecInfo.Var(:,iSLA) = 0.0001 * RecInfo.Var(:,iLA) ./ RecInfo.Var(:,iDM);

%create header
TXTh=[];
for i=1:length(RecInfo.StringNames) 
    TXTh=[TXTh sprintf('%s ,',char(RecInfo.StringNames{i}))];
end;
for i=1:length(RecInfo.NumNames) 
    TXTh=[TXTh sprintf('%s ,',char(RecInfo.NumNames{i}))]; 
end;
for i=1:length(RecInfo.VarNames) 
    TXTh=[TXTh sprintf('%s ,',char(RecInfo.VarNames{i}))]; 
end;

%Save header to file in the processed folder
fid=fopen(Outputfile,'w');
fprintf(fid,'%s \n', TXTh);

%add data
for i=1:RecInfo.NRecords
    TXT=[];
    for j=1:length(RecInfo.StringNames) 
        TXT=[TXT sprintf('%7s,',char(RecInfo.String{j}{i}))];
    end;
    for j=1:length(RecInfo.NumNames) 
        TXT=[TXT sprintf('%7f,',RecInfo.Num(i,j))]; 
    end;
    for j=1:length(RecInfo.VarNames) 
        TXT=[TXT sprintf('%7f,',RecInfo.Var(i,j))] ;
    end;
    fprintf(fid,'%s \n',TXT(1:end-1));
end  
fclose(fid);

%display data in combined figure, different colours per experiment
if Display
    col=get_graph_colours(1);
    iEXP=find(strcmp(RecInfo.StringNames, 'EXPERIMENT'));
    uEXP=unique(RecInfo.String{iEXP});
    h=[];NR=[];
    for i=1:length(uEXP)
        ii=logical(strcmp(uEXP(i),RecInfo.String{iEXP}));
        h=combofig(h,2,2,1,[],RecInfo.Var(ii,iDM),RecInfo.Var(ii,iDMc) ,col(i),RecInfo.VarNames{iDM},RecInfo.VarNames{iDMc} , '',0,2,2);
        h=combofig(h,2,2,2,[],RecInfo.Var(ii,iSLA),RecInfo.Var(ii,iDMc),col(i),RecInfo.VarNames{iSLA},RecInfo.VarNames{iDMc}, '',0,2,2);
        h=combofig(h,2,2,3,[],RecInfo.Var(ii,iDM),RecInfo.Var(ii,iLAI),col(i),RecInfo.VarNames{iDM},RecInfo.VarNames{iLAI}, '',0,2,2);
        [h, NR(i)]=combofig(h,2,2,4,[],RecInfo.Var(ii,iLAI),RecInfo.Var(ii,iSLA) ,col(i),RecInfo.VarNames{iLAI},RecInfo.VarNames{iSLA} , '',0,2,2);
    end
    legend(NR,uEXP)
    %save figure in PROCESSED folder
    saveas(h,[OutputFigure '.emf'],'emf');
    saveas(h,[OutputFigure '.eps'],'eps');
end





